<?php
namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function retrieve_users()
    {
        $users = User::all();
        return response()->json($users);
    }
    public function getRoles()
    {
        $roles = Role::all();
    
        return response()->json([
            'message' => 'Roles retrieved successfully',
            'roles' => $roles
        ], 200);
    }
      
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string',
            'phone_number' => 'required|string',
            'role_id' => 'required|integer|exists:roles,id',
        ]);
    
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'phone_number' => $request->phone_number,
            'status' => 1, // Set status to true (1) by default
            'role_id' => $request->role_id,
        ]);
    
        return response()->json([
            'message' => 'User registered successfully',
            'user' => $user
        ], 201);
    }
    
    
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $user->update([
            'name' => $request->name,
            'email' => $request->email,
            'password' => $request->password ? Hash::make($request->password) : $user->password,
        ]);
    
        return response()->json($user);
    }
    
  

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);
    
        $user = User::where('email', $request->email)->first();
    
        if (!$user || !Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages(['email' => 'Invalid credentials']);
        }
    
        return response()->json([
            'token' => $user->createToken('API Token')->plainTextToken,
            'user' => [
                'user_id' =>$user->id,
                'name' => $user->name,
                'email' => $user->email
            ]
        ]);
    }
    

    public function logout(Request $request)
    {
        $request->user()->tokens()->delete();
        return response()->json(['message' => 'Logged out successfully']);
    }
}
